/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx.c,v 1.72 2006/11/30 18:17:47 gallatin Exp $";

#include "mx_arch.h"
#include "mx_misc.h"
#include "mx_instance.h"
#include "mx_malloc.h"
#include "mx_pio.h"
#include "mx_peer.h"
#include "mx_version.h"

#include <miscfs/devfs/devfs.h>  
#include <sys/conf.h>   
#include <sys/fcntl.h>   
#include <sys/sysctl.h>

int mx_major = -1;
extern mx_endpt_state_t **mx_endpts;
void mx_munmapall(mx_endpt_state_t *es);
int mx_mmap(mx_endpt_state_t *es, mx_uaddr_t arg);

int
mx_rand(void)
{
  int val = (int)random();
  return val;
}


#if MX_DARWIN_XX == 7
static void 
proc_selfname(char * buf, int size)
{
  int len;
  struct proc *p = current_proc();

  if (sizeof(p->p_comm) > size)
    len = size;
  else
    len = sizeof(p->p_comm);

  bcopy(p->p_comm, buf, len);
}

#define proc_find pfind
#define uio_resid(x) x->uio_resid
#define uio_offset(x) x->uio_offset

#endif


/* It is normally safer to use IOMallocContiguous for dma page
   allocation, as we can always be certain of the alignment, but there
   is a memory leak in IOMallocContiguous() on Intel which exhausts
   kernel memory and panics the machine after a while */

#if !MX_CPU_x86 && !MX_CPU_x86_64
#define MX_USE_IOMALLOCCONTIGUOUS 1
#endif 


/*
  We need to use our own malloc because the OSX malloc allocates from
  a tiny (16MB) kalloc_map submap for large allocations.  To get
  around this, we must call IOMallocAligned, which in turns calls
   kernel_memory_allocate() so that we can specify that we want our
  allocation to come from the larger kernel_map.  For smaller
  allocations, we can call the Mach kalloc directly (for small
  allocations, it uses the zone allocator, and that grabs memory from
  the kernel_map, not the kmem_alloc submap).

  All of this means that when we free memory, we must know what the
  size of the allocation was.

*/

struct mx_macos_mem_hint {
  unsigned long len;
#if MX_DEBUG
  unsigned long cookie;
#endif
  int index;
  char buffer[0];
};

void *
mx_kmalloc(size_t len, uint32_t flags)
{
  size_t alloc_len;
  struct mx_macos_mem_hint *hint = 0;

  alloc_len = len + sizeof(*hint);

  hint = IOMallocAligned(alloc_len, 1);
  if (!hint) {
      MX_WARN(("kalloc of %ld bytes failed, nowait = %d \n", 
	       alloc_len, flags & MX_NOWAIT));
      return NULL;
  }

  if (flags & MX_MZERO)
    bzero(hint, alloc_len);
  
  hint->len = alloc_len;
#if MX_DEBUG
  hint->cookie = 0xfeedface;
#endif
  hint++;
  return hint;
}

void
mx_kfree(void *ptr)
{
  struct mx_macos_mem_hint *hint;
  unsigned long len;

  hint = (struct mx_macos_mem_hint *)ptr;
  hint--;
  len = hint->len;

#if MX_DEBUG
  if (hint->cookie != 0xfeedface) {
    IOLog("mx_kfree got bad cookie %lx when feeing %ld bytes\n",
	  hint->cookie, hint->len);
    panic("Bad MX free");
  }
  hint->cookie = 0;
  memset(hint, 0xff, len);	
#endif
  IOFreeAligned(hint, len);

}


#ifdef MX_USE_IOMALLOCCONTIGUOUS
int
mx_alloc_dma_page(mx_instance_state_t *is, char **alloc_addr, 
		  char **addr, mx_page_pin_t *pin)
{
  int status;
  IOPhysicalAddress dma;

  *alloc_addr = *addr = IOMallocContiguous(PAGE_SIZE, MX_VPAGE_SIZE,
					   &dma);
  if (!*addr)
    return ENOMEM;
  mx_assert(((uintptr_t)*addr & MX_VPAGE_MASK) == (uintptr_t)*addr);
  pin->va = (uint64_t)(uintptr_t)*addr;
  pin->dma.low = dma;
  pin->dma.high = 0;
  return 0;
}

void
mx_free_dma_page(mx_instance_state_t *is, char **alloc_addr, mx_page_pin_t *pin)
{
  IOFreeContiguous(*alloc_addr, PAGE_SIZE);
  if (MX_DEBUG)
    *alloc_addr = 0;
}

#else
int
mx_alloc_dma_page(mx_instance_state_t *is, char **alloc_addr, 
		  char **addr, mx_page_pin_t *pin)
{
  int status;

  *alloc_addr = *addr = (char *)IOMalloc(PAGE_SIZE);
  if (!*addr)
    return ENOMEM;  
  mx_always_assert(((uintptr_t)*addr & MX_VPAGE_MASK) == (uintptr_t)*addr);
  pin->va = (uint64_t)(uintptr_t)*addr;
  status = mx_pin_page(is, pin, MX_PIN_KERNEL | MX_PIN_CONSISTENT, 0);
  if (status) {
    IOFree((void *)*addr, PAGE_SIZE);
  }
  return status;
}

void
mx_free_dma_page(mx_instance_state_t *is, char **alloc_addr, mx_page_pin_t *pin)
{
  mx_unpin_page(is, pin, MX_PIN_KERNEL | MX_PIN_CONSISTENT);
  IOFree((void *)*alloc_addr, PAGE_SIZE);
  if (MX_DEBUG)
    *alloc_addr = 0;
}


#endif /* MX_USE_IOMALLOCCONTIGUOUS */



/* initialize the sync structure */

void
mx_sync_init(mx_sync_t *s, mx_instance_state_t *is, int endpoint, char *string)
{
  s->cookie = 0xdeadbeef;
  s->mtx = IOLockAlloc();
  s->sleep_mtx = IOLockAlloc();
  s->wake_cnt = 0;
}

void
mx_sync_destroy(mx_sync_t *s)
{
  IOLockFree(s->mtx);
  s->mtx = NULL;
  IOLockFree(s->sleep_mtx);
  s->sleep_mtx = NULL;
  s->wake_cnt = 0;
  s->cookie = 0;
}

void
mx_sync_reset(mx_sync_t *s)
{
  IOLockLock(s->sleep_mtx);
  s->wake_cnt = 0;
  IOLockUnlock(s->sleep_mtx);
}


void
mx_wake(mx_sync_t * s)
{
  mx_assert(s->cookie == 0xdeadbeef);
  IOLockLock(s->sleep_mtx);
  s->wake_cnt++;
  IOLockUnlock(s->sleep_mtx);
  thread_wakeup_one(s);
}


int
mx_sleep(mx_sync_t *s, int ms, int flags)
{
  int ret = THREAD_AWAKENED;
  AbsoluteTime deadline;
  int intr;

  if (s->cookie != 0xdeadbeef)
    panic("mx_sleep got bad cookie");

  if (ms != MX_MAX_WAIT) {
    clock_interval_to_deadline(ms, 1000*NSEC_PER_USEC, 
#if MX_DARWIN_XX >= 8
			       (uint64_t *)
#endif
			       &deadline);
  }

  intr = (flags & MX_SLEEP_INTR) ? THREAD_ABORTSAFE : THREAD_UNINT;
  IOLockLock(s->sleep_mtx);
  while (ret == 0 && s->wake_cnt <= 0) {
    if (ms != MX_MAX_WAIT)
    ret = IOLockSleepDeadline(s->sleep_mtx, s, deadline, intr);
      else
    ret = IOLockSleep(s->sleep_mtx, s, intr);
  }
  
  if (ret == THREAD_AWAKENED)
    s->wake_cnt--;

  IOLockUnlock(s->sleep_mtx);

  if (ret != THREAD_AWAKENED) {
    if (ret == THREAD_TIMED_OUT)
      return EAGAIN;
    else
      return EINTR;
  }
  return 0;
}

void
mx_spin_lock_init(mx_spinlock_t *s, mx_instance_state_t *is, int endpoint, char *string)
{
  *s = IOSimpleLockAlloc();
  mx_always_assert(*s != NULL);
}

void
mx_spin_lock_destroy(mx_spinlock_t *s)
{
  IOSimpleLockFree(*s);
  *s = NULL;
}

void 
mx_assertion_failed (const char *assertion, int line, const char *file)
{
  IOLog("MX: assertion: <<%s>>  failed at line %d, file %s\n",
	 assertion, line, file);
  IOLog("Don't panic..\n");
}

void *
mx_map_io_space(mx_instance_state_t * is, uint32_t offset, uint32_t len)
{
  return ((void *) ((vm_offset_t) is->arch.csr + (vm_offset_t) offset));
}

void
mx_unmap_io_space (mx_instance_state_t * is,
                        uint32_t len, void *kaddr)
{
}

/* deal with open/close/ioctl here */

static int mx_open(dev_t dev, int flags, int devtyple, struct proc *p);
static int mx_close(dev_t dev, int flags, int devtyple, struct proc *p);
static int mx_ioctl(dev_t dev, u_long cmd, caddr_t data, int fflag,
                           struct proc *p);
static int mx_read(dev_t dev, struct uio *uio, int ioflag);
static int mx_select(dev_t dev, int which, void * wql, struct proc *p);

static struct cdevsw mx_cdevsw = {
  mx_open,                /* d_open */
  mx_close,               /* d_close */
  mx_read,                /* d_read */
  eno_rdwrt,              /* d_write */
  mx_ioctl,               /* d_ioctl */
  eno_stop,               /* d_stop */
  eno_reset,              /* d_reset */
  0,                      /* d_ttys */
  mx_select,		  /* d_select */
  eno_mmap,               /* d_mmap */
  eno_strat,              /* strategy */
  eno_getc,               /* d_getc */
  eno_putc,               /* d_putc */
  0                       /* d_type */
};

static inline int
mx_minor_to_board(int i)
{
  int board;

  if (i < 4 * mx_max_instance)
    return i/4;

  i -= 4 * mx_max_instance;
  board = i / mx_max_endpoints;
  return board;
}

static inline int
mx_minor_to_endpt(int board, int i)
{
  i -= 4 * mx_max_instance;
  i -= board * mx_max_endpoints;
  return i;
}

int
mx_special_create(void)
{
  int i, perms;
  char name[16];

  mx_major = cdevsw_add(-1, &mx_cdevsw);
  if (mx_major == -1)
    {
      return mx_major;
    }
  /* got a cdevsw_entry, now create device nodes via devfs */

  for(i = 0; i < mx_max_minor; i++) {
    if (i < 4 * mx_max_instance) {
      switch (i & 0x3) {
      case 0:
	sprintf(name, "mx%d", i / 4);
	perms = 0666;
	break;
      case 1:
	/* priv. port, so use secure permissions */
	sprintf(name, "mxp%d", i / 4);
	perms = 0600;
	break;
      case 3:
	/* priv. raw port, so use secure permissions */
	sprintf(name, "mxr%d", i / 4);
	perms = 0600;
	break;
      default:
	/* would be unpriv raw port, but we don't want 2 raw ports */
	continue;
	break;
      }
    } else {
      int board = mx_minor_to_board(i);
      int endpt = mx_minor_to_endpt(board, i);
      sprintf(name, "mx%de%d", board, endpt);
      perms = 0666;
    }

    /* create the special file */
    mx_specials[i] = devfs_make_node(makedev(mx_major, i),
				     DEVFS_CHAR, UID_ROOT, GID_WHEEL,
				     perms, name, i);
    if(mx_specials[i] == NULL) {
      mx_special_destroy();
      return -1;
    }
  }
  mx_ctl[0] = devfs_make_node(makedev(mx_major, MX_CTL), DEVFS_CHAR,
			      UID_ROOT, GID_WHEEL, 0666, "mxctl", MX_CTL);
  mx_ctl[1] = devfs_make_node(makedev(mx_major, MX_CTLP), DEVFS_CHAR,
			      UID_ROOT, GID_WHEEL, 0600, "mxctlp", MX_CTLP);
  if (mx_ctl[0] == NULL || mx_ctl[1] == NULL) {
    mx_special_destroy();
    return -1;
  }
  return mx_major;
}

void
mx_cdevsw_remove(void)
{
  (void)cdevsw_remove(mx_major, &mx_cdevsw);
}

void
mx_special_destroy(void)
{
  int i;

  for (i = 0; i < mx_max_minor; i++)
    if (mx_specials[i] != NULL) {
          devfs_remove(mx_specials[i]);
          mx_specials[i] = 0;
    }
  if (mx_ctl[0] != NULL) {
    devfs_remove(mx_ctl[0]);
    mx_ctl[0] = 0;
  }
  if (mx_ctl[1] != NULL) {
    devfs_remove(mx_ctl[1]);
    mx_ctl[1] = 0;
  }
  mx_major = -1;
}

/*
 * Certain devices cannot have endpoints associated with them.  These
 * devices include /dev/mx{,p}0 (minors 0 and 1), /dev/mx{,p}1 (minors
 * 4 and 5), through /dev/mx{,p}n (minors 4*n and 4*n + 1), and
 * /dev/mxctl{,p} (minors 254 and 255).
 *
 * However, the "raw" devices, and well as the normal endpoints
 * all must have endpoint state associated with them
 */
#define MX_ENDPTLESS(x) (\
  ((x < 4 * mx_max_instance) && ((x % 4 == 0) || (x % 4 == 1))) 	\
  || x == MX_CTL || x == MX_CTLP)

static int 
mx_open(dev_t dev, int flags, int devtyple, struct proc *p)
{
  int minor, status;
  mx_endpt_state_t *es;

  minor = minor(dev);
  if (MX_ENDPTLESS(minor)) {
    return 0;
  }
  if (minor >= mx_max_minor)
    return ENODEV;

  if (mx_endpts[minor])
    return EBUSY;

  es = mx_kmalloc(sizeof(*es), MX_MZERO|MX_WAITOK);
  if (es == 0)
    return ENOMEM;

  SLIST_INIT(&es->arch.mmapmd);
  mx_mutex_enter(&mx_global_mutex);
  if (mx_endpts[minor]) {
    mx_mutex_exit(&mx_global_mutex);
    mx_kfree(es);
    return EBUSY;
  } else {
    status = 0;
    mx_endpts[minor] = es;
  }
  mx_mutex_exit(&mx_global_mutex);

  return status;
}

static void
mx_close_body(mx_endpt_state_t *es)
{
  /* only close the endpoint if its open.  An
     open endpoint will have an instance associated with it*/
  if (es->is) {
    mx_munmapall(es);
    mx_common_close(es);
  }
  mx_kfree(es);
}

static int 
mx_close(dev_t dev, int flags, int devtyple, struct proc *p)
{
  int minor;
  mx_endpt_state_t *es;

  minor = minor(dev);
  if (minor >= mx_max_minor)
    return ENODEV;
  mx_mutex_enter(&mx_global_mutex);
  es = mx_endpts[minor];
  mx_endpts[minor] = NULL;

  mx_mutex_exit(&mx_global_mutex);
  if (es)
    mx_close_body(es);

  return 0;
}

void    psignal __P((struct proc *p, int sig));

/* iterate over all open endpoints, looking for endpoints which
   are "stuck" */

static void
mx_endpoint_reaper(int force)
{
  int minor;
  mx_endpt_state_t *es;
  struct proc *p;
  boolean_t funnel_state;
  char *msg;
  
  mx_mutex_enter(&mx_global_mutex);
  for (minor = 0; minor < mx_max_minor; minor++) {
    es = mx_endpts[minor];
    if (es == NULL || es->is == NULL)
      continue;
    if (((p = proc_find(es->opener.pid)) == NULL) || force) {
      /*
       * if we cannot find the process in the process table,
       * that means that it is exiting & may be blocked trying
       * to free registered memory.  So give it some help..
       */
      if (force)
	msg = "still running at module unload";
      else
	msg = "stuck exiting";	
      MX_WARN(("mx%d: PID %d (%p) is %s, cleaning up es %p\n", 
	       es->is->id, es->opener.pid, p, msg, es));

      /* Kill the process -- this is the simplest way
	 to get rid of the progression thread.  In order to
         kill something, we must take the BSD kernel lock */
      if (p != NULL) {
#if MX_DARWIN_XX == 7
	funnel_state = thread_funnel_set(kernel_flock, TRUE);
	p = proc_find(es->opener.pid);
	if (p != NULL)
	  psignal(p, SIGKILL);
	(void)thread_funnel_set(kernel_flock, funnel_state);
#else
	proc_signal(es->opener.pid, SIGKILL);
	proc_rele(p);
#endif
      }
      /* Sleep for a second to see what happened.. Things may
	 have cleaned themselves up */
      mx_mutex_exit(&mx_global_mutex);
      IOSleep(1000);
      mx_mutex_enter(&mx_global_mutex);

      /* If we're still stuck, close the endpoint by hand */
      if (es == mx_endpts[minor]) {
	mx_endpts[minor] = 0;
	mx_mutex_exit(&mx_global_mutex);
	mx_close_body(es);
	mx_mutex_enter(&mx_global_mutex);
      }
    }
  }
  mx_mutex_exit(&mx_global_mutex);
}


static volatile int mx_timeout_thread_should_exit = 0;
static volatile int mx_timeout_thread_exited = 0;

static void
mx_timeout_thread(void *dontcare)
{
  do {
    mx_endpoint_reaper(0);
    mx_watchdog_body();
    IOSleep(2 * MX_WATCHDOG_TIMEOUT * 1000);
  } while (!mx_timeout_thread_should_exit);

  mx_timeout_thread_exited = 1;
  IOExitThread();
}

void
mx_start_timeout_thread(void)
{
  IOCreateThread(mx_timeout_thread, NULL);
}

void
mx_closeall(void)
{
  mx_endpoint_reaper(1);
}

void
mx_stop_timeout_thread(void)
{
  int count = 0;

  mx_timeout_thread_should_exit = 1;
  while(!mx_timeout_thread_exited && count < 5) {
    IOSleep(MX_WATCHDOG_TIMEOUT * 1500);
    IOLog("mx: waiting for timeout thread to exit\n");
    count++;
  }
  if (mx_timeout_thread_exited)
    IOLog("mx: timeout thread exited\n");
  else
    IOLog("mx: timeout thread blocked.. reboot now\n");
  
  /* sleep an extra 1.0s to give IOExitThread() a chance to really
     exit */
  IOSleep(1000);
}

static int
mx_is_64b(void)
{
#if MX_DARWIN_XX <= 7
  return 0;
#else
  int is_64b;
  proc_t p;

  p = proc_self();
  is_64b = proc_is64bit(p);
#if MX_DARWIN_XX != 9
  proc_rele(p);
#endif
  return is_64b;
#endif  
}

static int
mx_set_endpoint(mx_endpt_state_t *es, int minor, mx_uaddr_t addr, int raw)
{  
  int board, endpt, status;
  mx_set_endpt_t set_endpt;
  size_t len;

  if (minor >= mx_max_minor)
    return ERANGE;

  if (!mx_endpts[minor])
    return ENXIO;

  board = mx_minor_to_board(minor);
  endpt = mx_minor_to_endpt(board, minor);

  if (!raw) {
    status = mx_arch_copyin(addr, &set_endpt, sizeof(set_endpt));
    if (status)
      return status;

    /* make sure we're trying to set the same endpoint we were opened
       with.  Otherwise, the driver will be hopelessly confused */

    if (set_endpt.endpoint != endpt)
      return ENXIO;
  }

  es->opener.pid = mx_kgetpid();
  es->privileged = minor & 1;
  es->is_kernel = 0;
  es->arch.is_64b = mx_is_64b();
  es->arch.task = current_task();
  status = mx_common_open(board, set_endpt.endpoint, es, raw);
  proc_selfname(es->opener.comm, sizeof(es->opener.comm));
  set_endpt.session_id = es->session_id;

  if (status != 0) {
    es->is = 0;
    return (status);
  }
  
  if (!raw)
    status = mx_arch_copyout(&set_endpt, addr, sizeof(set_endpt));
  return status;
}

static int 
mx_ioctl(dev_t dev, u_long cmd, caddr_t data, int fflag, struct proc *p)
{
  int minor, privileged, retval, endptless, is_64b, meta_minor;
  mx_endpt_state_t *es;
  mx_uaddr_t arg;

  meta_minor = 0;
  minor = minor(dev);

  if (MX_ENDPTLESS(minor)) {
    meta_minor = 1;
    endptless = 1;
    es = NULL;
  } else if (minor < mx_max_minor) {
    mx_mutex_enter(&mx_global_mutex);
    es = mx_endpts[minor];
    mx_mutex_exit(&mx_global_mutex);
    endptless = 0;
  } else {
    return ENODEV;
  }    
  privileged = minor & 1;

  if (endptless == 0 && cmd != MX_SET_ENDPOINT) {
    is_64b = es->arch.is_64b;
  } else {
    is_64b = mx_is_64b();
  }

  if (data) {
    if (!is_64b) {
      arg = (mx_uaddr_t)(*(uintptr_t *)data);
    } else {
      arg = (mx_uaddr_t)(*(mx_uaddr_t *)data);
    }
  }
  else
    arg = 0;

  if (endptless) {
      retval = mx_endptless_ioctl(cmd, arg, privileged, 0);
  } else {
    /* if we have an endpoint but it is not setup yet, ensure
       that it is set up before allowing other ioctls */
    if (es->is == NULL) {
      if (cmd == MX_SET_ENDPOINT) {
	retval = mx_set_endpoint(es, minor, arg, 0);
      } else if (cmd == MX_SET_RAW) {
	retval = mx_set_endpoint(es, minor, arg, 1);
      } else {
	retval = EINVAL;
      }
      goto out;
    }
    mx_mutex_enter(&es->sync);
    es->ref_count++;
    mx_mutex_exit(&es->sync);
    if (cmd == MX_MMAP) {
      retval = mx_mmap(es, arg);
    } else {
      retval = mx_common_ioctl(es, cmd, arg);
      if (retval == ENOTTY) 
	retval = mx_endptless_ioctl(cmd, arg, privileged, 0);
    }
    mx_mutex_enter(&es->sync);
    es->ref_count--;
    mx_mutex_exit(&es->sync);  
  }
 out:

  /*  IOLog("%d bit ioctl 0x%lx at 0x%llx returns %d\n", 
      32 + 32 * is_64b, cmd, (long long)arg, retval); */
  return(retval);
}

static int
mx_read(dev_t dev, struct uio *uio, int ioflag)
{
  int len = 0;
  int resid;
  int status = 0;  
  int minor = minor(dev);
  int unit;
  off_t offset;
  char *c = 0;

  if (minor >= mx_max_minor)
    return ENODEV;
  unit = mx_minor_to_board(minor);

  status = mx_instance_status_string(unit, &c, &len);
   if (status)
    return status;

  resid = uio_resid(uio);
  offset = uio_offset(uio);
  if (offset > strlen(c))
      goto abort_with_c;
  status = uiomove(c+offset, MIN((len - offset), resid), uio);

 abort_with_c:
  mx_kfree(c);
  return status;

}

static int
mx_select(dev_t dev, int which, void * wql, struct proc *p)
{
  mx_endpt_state_t *es;
  mx_instance_state_t *is;
  unsigned long flags;
  int ready = 0;

  /* Only respond to read requests */
  if (which != FREAD) {
    return 0;
  }

  es = mx_endpts[minor(dev)];
  if (es == NULL) {
    return ready;
  }
  
  if (es->flags != MX_ES_RAW) {
    return ready;
  }

  is = es->is;

  /* check to see if the mcp died, since the raw endpoint opener
     will want to know about it */
  if (mx_is_dead(is))
    return ready;  

  mx_spin_lock_irqsave(&is->raw.spinlock, flags);
  if (STAILQ_FIRST(&is->raw.pending_events)) {
    ready = 1;
  } else {
    selrecord(p, is->arch.raw_si, wql);
    is->raw.wakeup_needed = 1;
  }
  mx_spin_unlock_irqrestore(&is->raw.spinlock, flags);

  return ready;
}

/* Hook to be able to reboot machine via remote gdb */
#include <IOKit/IOPlatformExpert.h>
void  mx_reboot(void);

void 
mx_reboot()
{
  (*PE_halt_restart)(kPERestartCPU);
}


void
mx_get_memory_size(uint64_t *bytes)
{
#if  MX_DARWIN_XX <= 7
  *bytes = max_mem;
#else
  int err;
  size_t size;

  size = sizeof(*bytes);
  err = sysctlbyname("hw.memsize", bytes, &size, NULL, 0);
  if (err != 0) {
    MX_WARN(("Could not find memory size, assuming 512MB to be safe\n"));
    *bytes = 512 * 1024 * 1024;
  }
#endif
}


#if  MX_DARWIN_XX <= 7
extern char hostname[];

void
mx_set_default_hostname(void)
{
  strncpy(mx_default_hostname, hostname, sizeof(mx_default_hostname) - 1);
  mx_default_hostname[sizeof(mx_default_hostname) - 1] = '\0';
}
#else
void
mx_set_default_hostname(void)
{
  strncpy(mx_default_hostname, "localhost", sizeof(mx_default_hostname) - 1);
  mx_default_hostname[sizeof(mx_default_hostname) - 1] = '\0';
}

#endif

void
mx_ether_tx_done(mx_instance_state_t *is, uint32_t mcp_index)
{
  is->arch.ether_tx_done((struct mx_instance_state *)is, mcp_index);
}

void
mx_ether_rx_done_small(mx_instance_state_t *is, int count, int len, 
		       int csum, int flags)
{
  is->arch.ether_rx_done_small((struct mx_instance_state *)is, count, len, 
			       csum, flags);
}

void
mx_ether_rx_done_big(mx_instance_state_t *is, int count, int len, 
		     int csum, int flags)
{
  is->arch.ether_rx_done_big((struct mx_instance_state *)is, count, len, 
			     csum, flags);
}

void
mx_ether_link_change_notify(mx_instance_state_t *is)
{
  if (is->arch.ether_link_change != NULL)
    is->arch.ether_link_change(is);
}

int
mx_ether_parity_detach(mx_instance_state_t *is)
{
  MX_WARN(("mx_ether_parity_detach called!\n"));
  return 0;
}

void
mx_ether_parity_reattach(mx_instance_state_t *is)
{
  MX_WARN(("mx_ether_parity_reattach called!\n"));
}

extern kern_return_t _start (kmod_info_t * ki, void *data);
extern kern_return_t _stop (kmod_info_t * ki, void *data);

KMOD_EXPLICIT_DECL(com.Myricom.iokit.mx_driver, \
		   MX_SHORT_VERSION_STR,  _start, _stop)
        __private_extern__ kmod_start_func_t *_realmain = 0;
        __private_extern__ kmod_stop_func_t *_antimain = 0;
